<?php

class CommandHandler {
    
    private $bot;
    
    public function __construct(TelegramBot $bot) {
        $this->bot = $bot;
    }
    
    /**
     * Process incoming update
     */
    public function process() {
        $text = $this->bot->getText();
        $userId = $this->bot->getUserId();
        $userName = $this->bot->getUserFirstName();
        $chatId = $this->bot->getChatId();
        $chatType = $this->bot->getChatType();
        
        // Check if it's a command
        if (strpos($text, '/') === 0) {
            return $this->handleCommand($text, $userId, $userName, $chatId, $chatType);
        }
        
        // Check if it's a group mention (user mentions a group name)
        if ($chatType === 'group' || $chatType === 'supergroup') {
            return $this->handleGroupMention($text, $userId, $chatId);
        }
        
        return false;
    }
    
    /**
     * Handle bot commands
     */
    private function handleCommand($text, $userId, $userName, $chatId, $chatType) {
        // Only owner can use admin commands in private chat
        if ($chatType !== 'private' && $userId !== OWNER_ID) {
            return false;
        }
        
        $parts = explode(' ', trim($text));
        $command = strtolower($parts[0]);
        
        switch ($command) {
            case '/start':
                return $this->cmdStart($chatId);
                
            case '/help':
                return $this->cmdHelp($chatId);
                
            case '/addchick':
                // Use reply to add a chick
                return $this->cmdAddChick($chatId);
                
            case '/removechick':
                return $this->cmdRemoveChick($chatId);
                
            case '/listchicks':
                return $this->cmdListChicks($chatId);
                
            case '/add':
                // /add groupname - Add replied user to group
                if (count($parts) < 2) {
                    $this->bot->sendMessage($chatId, "❌ Usage: /add <group_name>");
                    return true;
                }
                return $this->cmdAddToGroup($chatId, $parts[1], $userId);
                
            case '/remove':
                // /remove groupname userid
                if (count($parts) < 3) {
                    $this->bot->sendMessage($chatId, "❌ Usage: /remove <group_name> <user_id>");
                    return true;
                }
                return $this->cmdRemoveFromGroup($chatId, $parts[1], $parts[2]);
                
            case '/groups':
                return $this->cmdListGroups($chatId);
                
            case '/groupmembers':
                if (count($parts) < 2) {
                    $this->bot->sendMessage($chatId, "❌ Usage: /groupmembers <group_name>");
                    return true;
                }
                return $this->cmdGroupMembers($chatId, $parts[1]);
                
            default:
                return false;
        }
    }
    
    /**
     * /start command
     */
    private function cmdStart($chatId) {
        $text = "🤖 <b>Welcome to Your Telegram Assistant Bot!</b>\n\n";
        $text .= "This bot helps you manage your group chats and mentions.\n\n";
        $text .= "Use /help to see all available commands.";
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * /help command
     */
    private function cmdHelp($chatId) {
        $text = "📋 <b>Available Commands:</b>\n\n";
        
        $text .= "<b>Chick Management:</b>\n";
        $text .= "/addchick - Reply to a message and use this to add user as 'chick'\n";
        $text .= "/removechick - Remove a chick from your list\n";
        $text .= "/listchicks - Show all your chicks\n\n";
        
        $text .= "<b>Group Management:</b>\n";
        $text .= "/add <group_name> - Reply to user to add them to a group\n";
        $text .= "/remove <group_name> <user_id> - Remove user from group\n";
        $text .= "/groups - List all your groups\n";
        $text .= "/groupmembers <group_name> - List members of a group\n";
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * /addchick command
     */
    private function cmdAddChick($chatId) {
        if (!$this->bot->isReply()) {
            return $this->bot->sendMessage($chatId, "❌ Please reply to a message from the user you want to add as a chick.");
        }
        
        $repliedMsg = $this->bot->getRepliedMessage();
        $userId = $repliedMsg['from']['id'];
        $userName = $repliedMsg['from']['first_name'];
        
        if (Storage::addChick($userId)) {
            return $this->bot->sendMessage($chatId, "✅ <b>$userName</b> has been added to your chicks list!");
        } else {
            return $this->bot->sendMessage($chatId, "⚠️ <b>$userName</b> is already in your chicks list.");
        }
    }
    
    /**
     * /removechick command
     */
    private function cmdRemoveChick($chatId) {
        $chicks = Storage::getChicks();
        if (empty($chicks)) {
            return $this->bot->sendMessage($chatId, "❌ You have no chicks in your list.");
        }
        
        return $this->bot->sendMessage($chatId, "⚠️ Please use /removechick <user_id> to remove a chick.");
    }
    
    /**
     * /listchicks command
     */
    private function cmdListChicks($chatId) {
        $chicks = Storage::getChicks();
        
        if (empty($chicks)) {
            return $this->bot->sendMessage($chatId, "📭 You have no chicks in your list yet.");
        }
        
        $text = "👯 <b>Your Chicks List:</b>\n\n";
        foreach ($chicks as $id) {
            $text .= "• User ID: <code>$id</code>\n";
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * /add command - Add user to group
     */
    private function cmdAddToGroup($chatId, $groupName, $userId) {
        if (!$this->bot->isReply()) {
            return $this->bot->sendMessage($chatId, "❌ Please reply to a message from the user you want to add to <b>$groupName</b>.");
        }
        
        $repliedMsg = $this->bot->getRepliedMessage();
        $targetUserId = $repliedMsg['from']['id'];
        $targetUserName = $repliedMsg['from']['first_name'];
        
        // Create group if it doesn't exist
        if (!Storage::groupExists($groupName)) {
            Storage::createGroup($groupName);
        }
        
        if (Storage::addToGroup($groupName, $targetUserId, $targetUserName)) {
            return $this->bot->sendMessage($chatId, "✅ <b>$targetUserName</b> has been added to group <b>$groupName</b>!");
        } else {
            return $this->bot->sendMessage($chatId, "⚠️ <b>$targetUserName</b> is already in group <b>$groupName</b>.");
        }
    }
    
    /**
     * /remove command - Remove user from group
     */
    private function cmdRemoveFromGroup($chatId, $groupName, $userId) {
        if (Storage::removeFromGroup($groupName, $userId)) {
            return $this->bot->sendMessage($chatId, "✅ User has been removed from group <b>$groupName</b>!");
        } else {
            return $this->bot->sendMessage($chatId, "❌ User not found in group <b>$groupName</b>.");
        }
    }
    
    /**
     * /groups command
     */
    private function cmdListGroups($chatId) {
        $groups = Storage::getGroups();
        
        if (empty($groups)) {
            return $this->bot->sendMessage($chatId, "📭 You have no groups yet.");
        }
        
        $text = "📊 <b>Your Groups:</b>\n\n";
        foreach ($groups as $groupName => $members) {
            $text .= "• <b>$groupName</b> (" . count($members) . " members)\n";
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * /groupmembers command
     */
    private function cmdGroupMembers($chatId, $groupName) {
        if (!Storage::groupExists($groupName)) {
            return $this->bot->sendMessage($chatId, "❌ Group <b>$groupName</b> does not exist.");
        }
        
        $members = Storage::getGroupMembers($groupName);
        
        if (empty($members)) {
            return $this->bot->sendMessage($chatId, "📭 Group <b>$groupName</b> has no members.");
        }
        
        $text = "👥 <b>Members of $groupName:</b>\n\n";
        foreach ($members as $id => $name) {
            $text .= "• $name (<code>$id</code>)\n";
        }
        
        return $this->bot->sendMessage($chatId, $text);
    }
    
    /**
     * Handle group mention (when user types a group name)
     */
    private function handleGroupMention($text, $userId, $chatId) {
        $groups = Storage::getGroups();
        
        foreach ($groups as $groupName => $members) {
            if (strtolower($text) === strtolower($groupName) && !empty($members)) {
                // Mention all members in the group
                $mentions = '';
                foreach ($members as $id => $name) {
                    $mentions .= "<a href=\"tg://user?id=$id\">$name</a> ";
                }
                
                $responseText = "👥 <b>$groupName:</b>\n" . $mentions;
                return $this->bot->sendMessage($chatId, $responseText);
            }
        }
        
        return false;
    }
}
